%% Description
% Plots figure 1b: the adhesion force as a function of \Gamma for
% different fixed H_\infty (2,5,10,20).
%
% Data has been calculated by solving eqn (14) for non-contact solutions 
% and (16)-(18) for contact solutions for \Gamma from 1-10^4 and H_\infty 
% taking the values 2,5,10,20

%% Paramter definitions
%
% G(k)      Values of \Gamma used.
%
% H(i)      Values of H_\infty used: 2,5,10,20.
%
% j         Denotes the branch of solutions: 
%                       1 = stable non-contact, 
%                       2 = unstable, 
%                       3 = stable contact. 
%
% F(i,j,k)  Dimensionless adhesion force in equilibrium when \Gamma = G(k),
%           H_\infty = H(i) on branch j. F=0 when no such solution exists.
% 
% col       Colour scheme for plotting

clear all;close all;
%% Load calculated data
load('figure2b.mat')

%% Plotting
% Set colour scheme -- parula
col = parula(4);

% Plot force as function \Gamma  for H_\infty = {2,5,10,20}
for k = 1:4
    
% This is what we would plot if not bothered about gaps between types
% of solutions:
%     loglog(G,F(:,1,k),'color',col(k,:),'linewidth',2), hold on
%     loglog(G,F(:,2,k),'color',col(k,:),'linewidth',2,'linestyle','--')
%     loglog(G,F(:,3,k),'color',col(k,:),'linewidth',4)  
    
if k==1
    % Need to join between contact and not
    % Find where F non-zero first/last for contact/not
    n1 = find(F(:,1,k),1,'last');
    n2 = find(F(:,3,k),1,'first');
    
    loglog([G(1:n1) mean([G(n1) G(n2)])],...
        [F(1:n1,1,k); mean([F(n1,1,k) F(n2,3,k)])],...
        'color',col(k,:),'linewidth',2), hold on
    loglog([mean([G(n1) G(n2)]) G(n2:end)],...
        [mean([F(n1,1,k) F(n2,3,k)]); F(n2:end,3,k)],...
        'color',col(k,:),'linewidth',4)
    
else
    
    % Need to join between types of solutions in plot
    % Find where F non-zero first/last for each type of solution
    n1 = find(F(:,1,k),1,'last');
    n2 = find(F(:,2,k),1,'last');
    n3 = find(F(:,2,k),1,'first');
    n4 = find(F(:,3,k),1,'first');
    
    loglog([G(1:n1) mean([G(n1) G(n2)])],...
        [F(1:n1,1,k); mean([F(n1,1,k) F(n2,2,k)])],...
        'color',col(k,:),'linewidth',2)
    loglog([mean([G(n3) G(n4)]) G(n3:n2) mean([G(n1) G(n2)])],...
        [mean([F(n3,2,k) F(n4,3,k)]); F(n3:n2,2,k); mean([F(n1,1,k) F(n2,2,k)])],...
        'color',col(k,:),'linewidth',2,'linestyle','--')
    loglog([mean([G(n3) G(n4)]) G(n4:end)],...
        [mean([F(n3,2,k) F(n4,3,k)]); F(n4:end,3,k)],...
        'color',col(k,:),'linewidth',4)
    
end
end

% Label axes
xlabel('$\Gamma$','interpreter','Latex')
ylabel('$F$','interpreter','Latex')

% Increase axis label size
set(findobj('type','axes'),'fontsize',20,'linewidth',2)

% Increase tick length
set(gca,'TickLength',2*get(gca,'TickLength'))